<?php
// config.php
session_start();

// === DEBUG (turn off in prod) ===
// ini_set('display_errors','0');
ini_set('display_errors','1');
error_reporting(E_ALL);

// === JSON/CORS ===
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

require_once __DIR__ . '/db.php';
// Make sure the connection speaks utf8mb4
if (method_exists($mysqli, 'set_charset')) { $mysqli->set_charset('utf8mb4'); }

// === Google Drive setup ===
$GOOGLE_API_KEY   = 'AIzaSyAwrcIg1dVDhxwxJ9sWXy8-puwoREP0-OE'; // DO NOT REMOVE
$DRIVE_FOLDER_ID  = '1TTFcqpT10z8LiKfwlUOOv3Wn6Ciil6XW';

// ===== OAuth 2.0 (auto-refresh) FOR UPLOADS =====
$OAUTH_CLIENT_ID     = '35205627001-le4npdbfj97cai8k1koefhsjfhv17v2n.apps.googleusercontent.com';
$OAUTH_CLIENT_SECRET = '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';
$OAUTH_REFRESH_TOKEN = '1//04dwbbWZcvhcNCgYIARAAGAQSNwF-L9IrFf3qvpljUHt1wUcF4u6NiXawdFORk0koPQiDL5To-p_UBNVp371Bu7EFbrjqF6Qp1TE';

$GLOBALS['OAUTH_LAST_ERROR'] = null;

// ---------- Helpers ----------
function get_drive_access_token(): string {
  $GLOBALS['OAUTH_LAST_ERROR'] = null;
  $id     = $GLOBALS['OAUTH_CLIENT_ID']     ?? '';
  $secret = $GLOBALS['OAUTH_CLIENT_SECRET'] ?? '';
  $refresh= $GLOBALS['OAUTH_REFRESH_TOKEN'] ?? '';

  if ($id && $secret && $refresh) {
    if (!function_exists('curl_init')) {
      $GLOBALS['OAUTH_LAST_ERROR'] = 'cURL extension not available on server.';
    } else {
      $ch = curl_init('https://oauth2.googleapis.com/token');
      curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query([
          'client_id'     => $id,
          'client_secret' => $secret,
          'refresh_token' => $refresh,
          'grant_type'    => 'refresh_token',
        ]),
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
      ]);
      $body = curl_exec($ch);
      $err  = curl_error($ch);
      $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
      curl_close($ch);

      if ($body === false || $err) {
        $GLOBALS['OAUTH_LAST_ERROR'] = "cURL error contacting token endpoint: $err";
      } else {
        $j = json_decode($body, true);
        if (!empty($j['access_token'])) return $j['access_token'];
        $GLOBALS['OAUTH_LAST_ERROR'] = "Token response ($code): ".$body;
      }
    }
  }

  $env = getenv('GDRIVE_ACCESS_TOKEN') ?: '';
  if (!$env && !$GLOBALS['OAUTH_LAST_ERROR']) {
    $GLOBALS['OAUTH_LAST_ERROR'] = 'No OAuth client/refresh configured and GDRIVE_ACCESS_TOKEN env var is empty.';
  }
  return $env;
}

function wants_redirect(): bool { return isset($_POST['redirect']) && $_POST['redirect'] === '1'; }

function json_out($arr, $code=200) {
  if (!wants_redirect()) {
    http_response_code($code);
    echo json_encode($arr);
  }
  exit;
}

function save_profile_upload($field='profile_photo') {
  if (!isset($_FILES[$field]) || $_FILES[$field]['error'] !== UPLOAD_ERR_OK) return null;
  $dir = __DIR__ . '/uploads';
  if (!is_dir($dir)) mkdir($dir, 0755, true);
  $ext  = pathinfo($_FILES[$field]['name'], PATHINFO_EXTENSION);
  $safe = bin2hex(random_bytes(8)) . '.' . strtolower($ext ?: 'jpg');
  $dest = $dir . '/' . $safe;
  if (!move_uploaded_file($_FILES[$field]['tmp_name'], $dest)) return null;
  return 'uploads/' . $safe;
}

function drive_make_file_public($fileId, $accessToken) {
  if (!$fileId) return false;
  $url = "https://www.googleapis.com/drive/v3/files/$fileId/permissions";
  $payload = json_encode([
    'role' => 'reader',
    'type' => 'anyone',
    'allowFileDiscovery' => false
  ]);
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
      "Authorization: Bearer $accessToken",
      "Content-Type: application/json"
    ],
    CURLOPT_POSTFIELDS => $payload,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 60
  ]);
  curl_exec($ch);
  $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return ($code >= 200 && $code < 300);
}

/** Get an open cart id for the user, or create one */
function get_or_create_open_cart(int $user_id, mysqli $mysqli) {
  $stmt = $mysqli->prepare("SELECT id FROM carts WHERE user_id=? AND status='open' LIMIT 1");
  $stmt->bind_param('i', $user_id);
  $stmt->execute();
  $stmt->bind_result($cid);
  if ($stmt->fetch()) { $stmt->close(); return $cid; }
  $stmt->close();
  $stmt = $mysqli->prepare("INSERT INTO carts (user_id) VALUES (?)");
  $stmt->bind_param('i', $user_id);
  $stmt->execute();
  $id = $stmt->insert_id;
  $stmt->close();
  return $id;
}

// ---------- Router ----------
$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch ($action) {

  // ---------- Health ----------
  case 'ping': {
    json_out(['ok'=>true, 'php'=>'alive']);
    break;
  }

  case 'debug_oauth': {
    $t = get_drive_access_token();
    json_out([
      'ok' => (bool)$t,
      'has_token' => (bool)$t,
      'token_sample' => $t ? substr($t,0,12).'...' : null,
      'detail' => $GLOBALS['OAUTH_LAST_ERROR'],
      'curl_enabled' => function_exists('curl_init'),
      'allow_url_fopen' => (bool)ini_get('allow_url_fopen'),
    ]);
    break;
  }

  case 'debug_oauth_info': {
    json_out([
      'client_id_prefix' => substr($OAUTH_CLIENT_ID ?? '', 0, 24),
      'secret_set'       => !empty($OAUTH_CLIENT_SECRET),
      'refresh_len'      => strlen($OAUTH_REFRESH_TOKEN ?? ''),
    ]);
    break;
  }

  // ---------- Register (now with username) ----------
  case 'register': {
    $first = trim($_POST['first_name'] ?? '');
    $last  = trim($_POST['last_name'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $age   = intval($_POST['age'] ?? 0);
    $gender= $_POST['gender'] ?? '';
    $pass  = $_POST['password'] ?? '';
    $register_for = $_POST['register_for'] ?? 'User';

    if (!$first || !$last || !$username || !$email || !$age || !in_array($gender,['Male','Female'], true) || !$pass) {
      $msg = 'Missing/invalid fields';
      if (wants_redirect()) { header('Location: register.php?error='.rawurlencode($msg)); exit; }
      json_out(['ok'=>false,'error'=>$msg], 422);
    }

    // Enforce username uniqueness across BOTH tables
    foreach (['users','admins'] as $t){
      $st = $mysqli->prepare("SELECT 1 FROM $t WHERE username=? LIMIT 1");
      $st->bind_param('s',$username);
      $st->execute();
      if ($st->get_result()->fetch_row()){
        $st->close();
        $msg = 'Username already taken';
        if (wants_redirect()) { header('Location: register.php?error='.rawurlencode($msg)); exit; }
        json_out(['ok'=>false,'error'=>$msg], 409);
      }
      $st->close();
    }

    $photo_path = save_profile_upload('profile_photo');
    $password_hash = password_hash($pass, PASSWORD_DEFAULT);
    $table = (strcasecmp($register_for,'Admin')===0) ? 'admins' : 'users';

    $stmt = $mysqli->prepare("INSERT INTO $table
      (first_name,last_name,username,email,age,gender,password_hash,profile_photo_path)
      VALUES (?,?,?,?,?,?,?,?)");
    $stmt->bind_param('ssssisss', $first,$last,$username,$email,$age,$gender,$password_hash,$photo_path);

    try {
      $stmt->execute();
      if (wants_redirect()) { header('Location: login.php?registered=1'); exit; }
      json_out(['ok'=>true,'role'=> (strcasecmp($register_for,'Admin')===0)?'admin':'user','username'=>$username]);
    } catch (mysqli_sql_exception $e) {
      $msg = (stripos($e->getMessage(),'Duplicate')!==false)
          ? 'Email or username already registered in that role.'
          : $e->getMessage();
      if (wants_redirect()) { header('Location: register.php?error='.rawurlencode($msg)); exit; }
      json_out(['ok'=>false,'error'=>$msg], 409);
    } finally { $stmt->close(); }
    break;
  }

  // ---------- Login ----------
  case 'login': {
    $email = trim($_POST['email'] ?? '');
    $pass  = $_POST['password'] ?? '';

    if (!$email || !$pass) {
      $msg = 'Missing email or password';
      if (wants_redirect()) { header('Location: login.php?error='.rawurlencode($msg)); exit; }
      json_out(['ok'=>false,'error'=>$msg], 422);
    }

    $found = null; $role = null; $uid = null;
    foreach (['users'=>'user','admins'=>'admin'] as $tbl=>$r) {
      $stmt = $mysqli->prepare("SELECT id,username,password_hash,first_name,last_name,profile_photo_path FROM $tbl WHERE email=? LIMIT 1");
      $stmt->bind_param('s', $email);
      $stmt->execute();
      $res = $stmt->get_result();
      if ($row = $res->fetch_assoc()) {
        if (password_verify($pass, $row['password_hash'])) {
          $found = $row; $role = $r; $uid = intval($row['id']); $stmt->close(); break;
        }
      }
      $stmt->close();
    }

    if (!$found) {
      $msg = 'Invalid credentials';
      if (wants_redirect()) { header('Location: login.php?error='.rawurlencode($msg)); exit; }
      json_out(['ok'=>false,'error'=>$msg], 401);
    }

    $token = bin2hex(random_bytes(32));
    $stmt = $mysqli->prepare("INSERT INTO auth_tokens (email, role, token) VALUES (?,?,?)");
    $stmt->bind_param('sss', $email, $role, $token);
    $stmt->execute();
    $stmt->close();

    $_SESSION['email'] = $email;
    $_SESSION['role']  = $role;
    $_SESSION['name']  = trim(($found['first_name'] ?? '').' '.($found['last_name'] ?? ''));
    $_SESSION['username'] = $found['username'] ?? '';
    $_SESSION['user_id']  = $uid;

    if (wants_redirect()) { header('Location: index.php'); exit; }

    json_out([
      'ok'=>true,
      'token'=>$token,
      'role'=>$role,
      'user_id'=>$uid,
      'first_name'=>$found['first_name'],
      'last_name'=>$found['last_name'],
      'username'=>$found['username'],
      'profile_photo'=>$found['profile_photo_path']
    ]);
    break;
  }

  // ---------- Stats ----------
  case 'stats': {
    $u = $mysqli->query("SELECT COUNT(*) c FROM users")->fetch_assoc()['c'] ?? 0;
    $a = $mysqli->query("SELECT COUNT(*) c FROM admins")->fetch_assoc()['c'] ?? 0;
    json_out(['ok'=>true, 'users'=>intval($u), 'admins'=>intval($a)]);
    break;
  }

  // ---------- List people ----------
  case 'list': {
    $users = []; $admins = [];
    $ru = $mysqli->query("SELECT id,first_name,last_name,username,email,gender,age,profile_photo_path,created_at FROM users ORDER BY created_at DESC");
    while ($row = $ru->fetch_assoc()) $users[] = $row;
    $ra = $mysqli->query("SELECT id,first_name,last_name,username,email,gender,age,profile_photo_path,created_at FROM admins ORDER BY created_at DESC");
    while ($row = $ra->fetch_assoc()) $admins[] = $row;
    json_out(['ok'=>true, 'users'=>$users, 'admins'=>$admins]);
    break;
  }

  // ---------- Drive: list videos ----------
  case 'drive_list_videos': {
    $q = sprintf("'%s' in parents and mimeType contains 'video/' and trashed = false", $DRIVE_FOLDER_ID);
    $params = http_build_query([
      'q' => $q,
      'key' => $GOOGLE_API_KEY,
      'fields' => 'files(id,name,mimeType,webViewLink,webContentLink,thumbnailLink,videoMediaMetadata)'
    ]);
    $url = "https://www.googleapis.com/drive/v3/files?$params";
    $resp = @file_get_contents($url);
    if ($resp === false) json_out(['ok'=>false,'error'=>'Drive list failed'], 502);
    json_out(['ok'=>true,'data'=>json_decode($resp, true)]);
    break;
  }

  // ---------- Drive: upload video ----------
  case 'drive_upload': {
    $ACCESS_TOKEN = get_drive_access_token();
    if (!$ACCESS_TOKEN) {
      $detail = $GLOBALS['OAUTH_LAST_ERROR'] ?? 'Unknown';
      json_out(['ok'=>false,'error'=>'Missing Google Drive access token','detail'=>$detail], 400);
    }

    if (!isset($_FILES['video']) || $_FILES['video']['error'] !== UPLOAD_ERR_OK) {
      $err = $_FILES['video']['error'] ?? 'no-file';
      json_out(['ok'=>false,'error'=>'No video file (or upload error)','detail'=>$err], 422);
    }
    $fileName = basename($_FILES['video']['name']);
    $mime = mime_content_type($_FILES['video']['tmp_name']) ?: 'video/mp4';

    // 1) init session
    $meta = ['name'=>$fileName, 'parents'=>[$DRIVE_FOLDER_ID]];
    $ch = curl_init('https://www.googleapis.com/upload/drive/v3/files?uploadType=resumable');
    curl_setopt_array($ch, [
      CURLOPT_HTTPHEADER => [
        "Authorization: Bearer $ACCESS_TOKEN",
        "Content-Type: application/json; charset=UTF-8",
        "X-Upload-Content-Type: $mime"
      ],
      CURLOPT_POST => true,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_HEADER => true,
      CURLOPT_POSTFIELDS => json_encode($meta),
      CURLOPT_TIMEOUT => 60
    ]);
    $initResp = curl_exec($ch);
    if ($initResp === false) {
      json_out(['ok'=>false,'error'=>'Failed to init upload','detail'=>curl_error($ch)], 502);
    }
    $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
    $headers = substr($initResp, 0, $header_size);
    $status  = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $sessionUrl = null;
    if (preg_match('/^Location:\s*(.+)\r?$/im', $headers, $m)) { $sessionUrl = trim($m[1]); }
    if (!$sessionUrl) {
      json_out(['ok'=>false,'error'=>'Upload session URL not returned','status'=>$status,'headers'=>$headers], 502);
    }

    // 2) upload bytes
    $data = file_get_contents($_FILES['video']['tmp_name']);
    if ($data === false) {
      json_out(['ok'=>false,'error'=>'Failed to read uploaded file from temp dir'], 500);
    }
    $ch2 = curl_init($sessionUrl);
    curl_setopt_array($ch2, [
      CURLOPT_CUSTOMREQUEST => 'PUT',
      CURLOPT_HTTPHEADER => [
        "Authorization: Bearer $ACCESS_TOKEN",
        "Content-Length: " . strlen($data),
        "Content-Type: $mime"
      ],
      CURLOPT_POSTFIELDS => $data,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_TIMEOUT => 600
    ]);
    $finishResp = curl_exec($ch2);
    $err2  = curl_error($ch2);
    $code2 = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
    curl_close($ch2);

    if ($finishResp === false) {
      json_out(['ok'=>false,'error'=>'Upload failed','detail'=>$err2], 502);
    }
    if ($code2 < 200 || $code2 >= 300) {
      json_out(['ok'=>false,'error'=>'Drive API error','http'=>$code2,'details'=>$finishResp], $code2);
    }

    $file = json_decode($finishResp, true);
    drive_make_file_public($file['id'] ?? '', $ACCESS_TOKEN);

    json_out(['ok'=>true, 'file'=>$file], 200);
    break;
  }

  // ---------- Comments ----------
  case 'comment_add': {
    $file_id  = trim($_POST['file_id'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $role     = ($_POST['role'] ?? 'user') === 'admin' ? 'admin' : 'user';
    $text     = trim($_POST['text'] ?? '');
    if (!$file_id || !$username || !$text) json_out(['ok'=>false,'error'=>'Missing fields'], 422);

    $st = $mysqli->prepare("INSERT INTO comments (file_id,username,role,comment_text) VALUES (?,?,?,?)");
    $st->bind_param('ssss', $file_id,$username,$role,$text);
    $st->execute();
    $id = $st->insert_id;
    $st->close();

    json_out(['ok'=>true,'id'=>$id]);
    break;
  }

  case 'comment_list': {
    $file_id = trim($_GET['file_id'] ?? '');
    if (!$file_id) json_out(['ok'=>false,'error'=>'file_id required'], 422);
    $rows = [];
    $rs = $mysqli->prepare("SELECT id,username,role,comment_text,created_at FROM comments WHERE file_id=? ORDER BY id DESC LIMIT 200");
    $rs->bind_param('s',$file_id);
    $rs->execute();
    $res = $rs->get_result();
    while ($r = $res->fetch_assoc()) $rows[] = $r;
    $rs->close();
    json_out(['ok'=>true,'comments'=>$rows]);
    break;
  }

  // ---------- Reactions (likes) ----------
  case 'react_toggle': {
    $file_id  = trim($_POST['file_id'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $role     = ($_POST['role'] ?? 'user') === 'admin' ? 'admin' : 'user';
    if (!$file_id || !$username) json_out(['ok'=>false,'error'=>'Missing fields'], 422);

    // toggle
    $chk = $mysqli->prepare("SELECT id FROM reactions WHERE file_id=? AND username=? LIMIT 1");
    $chk->bind_param('ss',$file_id,$username);
    $chk->execute();
    $row = $chk->get_result()->fetch_assoc();
    $chk->close();

    if ($row) {
      $del = $mysqli->prepare("DELETE FROM reactions WHERE id=?");
      $del->bind_param('i',$row['id']);
      $del->execute();
      $del->close();
      $liked = false;
    } else {
      $ins = $mysqli->prepare("INSERT INTO reactions (file_id,username,role,reaction) VALUES (?,?,?, 'like')");
      $ins->bind_param('sss',$file_id,$username,$role);
      $ins->execute();
      $ins->close();
      $liked = true;
    }

    $cnt = $mysqli->prepare("SELECT COUNT(*) c FROM reactions WHERE file_id=?");
    $cnt->bind_param('s',$file_id);
    $cnt->execute();
    $c = $cnt->get_result()->fetch_assoc()['c'] ?? 0;
    $cnt->close();
    json_out(['ok'=>true,'liked'=>$liked,'count'=>intval($c)]);
    break;
  }

  case 'react_get': {
    $file_id  = trim($_GET['file_id'] ?? '');
    $username = trim($_GET['username'] ?? '');
    if (!$file_id) json_out(['ok'=>false,'error'=>'file_id required'], 422);

    $cnt = $mysqli->prepare("SELECT COUNT(*) c FROM reactions WHERE file_id=?");
    $cnt->bind_param('s',$file_id);
    $cnt->execute();
    $c = $cnt->get_result()->fetch_assoc()['c'] ?? 0;
    $cnt->close();

    $liked = false;
    if ($username) {
      $chk = $mysqli->prepare("SELECT 1 FROM reactions WHERE file_id=? AND username=? LIMIT 1");
      $chk->bind_param('ss',$file_id,$username);
      $chk->execute();
      $liked = (bool)$chk->get_result()->fetch_row();
      $chk->close();
    }
    json_out(['ok'=>true,'count'=>intval($c),'liked'=>$liked]);
    break;
  }

  // ================= BOOKING / CART APIs =================

  // List active destinations (for "Recommended")
  case 'destinations_list': {
    $q = $mysqli->query("SELECT id,title,city,country,thumbnail_url,price
                         FROM destinations WHERE is_active=1
                         ORDER BY created_at DESC LIMIT 100");
    $rows = [];
    while ($r = $q->fetch_assoc()) $rows[] = $r;
    json_out(['ok'=>true, 'destinations'=>$rows]);
    break;
  }

  // Add item to cart
  case 'cart_add': {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') json_out(['ok'=>false,'error'=>'POST required'],405);
    $user_id = intval($_POST['user_id'] ?? 0);
    $dest_id = intval($_POST['destination_id'] ?? 0);
    $qty     = max(1, intval($_POST['qty'] ?? 1));
    if (!$user_id || !$dest_id) json_out(['ok'=>false,'error'=>'Missing user_id/destination_id'],422);

    $stmt = $mysqli->prepare("SELECT price FROM destinations WHERE id=? AND is_active=1");
    $stmt->bind_param('i', $dest_id);
    $stmt->execute();
    $stmt->bind_result($price);
    if (!$stmt->fetch()) { $stmt->close(); json_out(['ok'=>false,'error'=>'Invalid destination'],400); }
    $stmt->close();

    $cart_id = get_or_create_open_cart($user_id, $mysqli);

    // merge if exists
    $stmt = $mysqli->prepare("SELECT id, qty FROM cart_items WHERE cart_id=? AND destination_id=?");
    $stmt->bind_param('ii', $cart_id, $dest_id);
    $stmt->execute();
    $stmt->bind_result($item_id, $old_qty);
    if ($stmt->fetch()) {
      $stmt->close();
      $new_qty = $old_qty + $qty;
      $u = $mysqli->prepare("UPDATE cart_items SET qty=? WHERE id=?");
      $u->bind_param('ii', $new_qty, $item_id);
      $u->execute();
      $u->close();
    } else {
      $stmt->close();
      $i = $mysqli->prepare("INSERT INTO cart_items (cart_id, destination_id, qty, unit_price) VALUES (?,?,?,?)");
      $i->bind_param('iiid', $cart_id, $dest_id, $qty, $price);
      $i->execute();
      $i->close();
    }

    json_out(['ok'=>true, 'cart_id'=>$cart_id]);
    break;
  }

  // Get current open cart
  case 'cart_get': {
    $user_id = intval($_GET['user_id'] ?? 0);
    if (!$user_id) json_out(['ok'=>false,'error'=>'user_id required'],422);

    $stmt = $mysqli->prepare("
      SELECT c.id AS cart_id, ci.id AS item_id, d.id AS destination_id,
             d.title, d.thumbnail_url, ci.qty, ci.unit_price
      FROM carts c
      JOIN cart_items ci ON ci.cart_id=c.id
      JOIN destinations d ON d.id=ci.destination_id
      WHERE c.user_id=? AND c.status='open'
      ORDER BY ci.added_at DESC
    ");
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $res = $stmt->get_result();
    $items=[]; $cart_id=null;
    while ($r = $res->fetch_assoc()) { $cart_id = $r['cart_id']; $items[] = $r; }
    $stmt->close();

    json_out(['ok'=>true,'cart_id'=>$cart_id,'items'=>$items]);
    break;
  }

  // Remove one cart item
  case 'cart_remove': {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') json_out(['ok'=>false,'error'=>'POST required'],405);
    $item_id = intval($_POST['item_id'] ?? 0);
    if (!$item_id) json_out(['ok'=>false,'error'=>'item_id required'],422);
    $d = $mysqli->prepare("DELETE FROM cart_items WHERE id=?");
    $d->bind_param('i',$item_id);
    $d->execute();
    $d->close();
    json_out(['ok'=>true]);
    break;
  }

  // Checkout -> create booking
  case 'cart_checkout': {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') json_out(['ok'=>false,'error'=>'POST required'],405);
    $user_id = intval($_POST['user_id'] ?? 0);
    if (!$user_id) json_out(['ok'=>false,'error'=>'user_id required'],422);

    // locate open cart
    $stmt = $mysqli->prepare("SELECT id FROM carts WHERE user_id=? AND status='open' LIMIT 1");
    $stmt->bind_param('i',$user_id);
    $stmt->execute();
    $stmt->bind_result($cart_id);
    if (!$stmt->fetch()) { $stmt->close(); json_out(['ok'=>false,'error'=>'No open cart'],400); }
    $stmt->close();

    $res = $mysqli->query("
      SELECT ci.destination_id, d.title, ci.qty, ci.unit_price
      FROM cart_items ci
      JOIN destinations d ON d.id=ci.destination_id
      WHERE ci.cart_id=".$cart_id
    );
    $items=[]; $total=0;
    while ($r = $res->fetch_assoc()) { $items[] = $r; $total += ($r['qty']*$r['unit_price']); }
    if (!$items) json_out(['ok'=>false,'error'=>'Cart is empty'],400);

    // booking header
    $b = $mysqli->prepare("INSERT INTO bookings (user_id,total_amount,status) VALUES (?,?, 'pending')");
    $b->bind_param('id',$user_id,$total);
    $b->execute();
    $booking_id = $b->insert_id;
    $b->close();

    // booking items
    $bi = $mysqli->prepare("INSERT INTO booking_items (booking_id,destination_id,title_snapshot,qty,unit_price) VALUES (?,?,?,?,?)");
    foreach ($items as $it) {
      $dest_id = (int)$it['destination_id'];
      $title   = (string)$it['title'];
      $qty     = (int)$it['qty'];
      $unit    = (float)$it['unit_price'];
      $bi->bind_param('iisid',$booking_id,$dest_id,$title,$qty,$unit);
      $bi->execute();
    }
    $bi->close();

    // close cart
    $mysqli->query("UPDATE carts SET status='checked_out' WHERE id=".$cart_id);

    json_out(['ok'=>true,'booking_id'=>$booking_id,'total'=>round($total,2)]);
    break;
  }

  // List bookings for a user
  case 'bookings_list': {
    $user_id = intval($_GET['user_id'] ?? 0);
    if (!$user_id) json_out(['ok'=>false,'error'=>'user_id required'],422);
    $q = $mysqli->query("SELECT id,status,total_amount,created_at FROM bookings
                         WHERE user_id=".$user_id." ORDER BY id DESC");
    $rows=[];
    while ($r = $q->fetch_assoc()) $rows[] = $r;
    json_out(['ok'=>true,'bookings'=>$rows]);
    break;
  }

  // ---------- Fallback ----------
  default:
    json_out(['ok'=>false,'error'=>'Unknown action: '.$action], 400);
}
